#!/bin/bash
# Copyright (c) Meta Platforms, Inc. and affiliates.
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

# run this script from the project root using `./scripts/build_docs.sh`

usage() {
  echo "Usage: $0 [-b] [-o]"
  echo ""
  echo "Build BoTorch documentation. Must be executed from root of BoTorch repository."
  echo ""
  echo "  -b   Build static version of documentation (otherwise start server)."
  echo "  -o   Only Docusaurus (skip Sphinx, tutorials). Useful when just make change to Docusaurus settings."
  echo ""
  exit 1
}

BUILD_STATIC=false
ONLY_DOCUSAURUS=false

while getopts 'bho' flag; do
  case "${flag}" in
    b)
      BUILD_STATIC=true
      ;;
    h)
      usage
      ;;
    o)
      ONLY_DOCUSAURUS=true
      ;;
    *)
      usage
      ;;
  esac
done

if [[ $ONLY_DOCUSAURUS == false ]]; then
  echo "-----------------------------------"
  echo "Generating API reference via Sphinx"
  echo "-----------------------------------"
  cd sphinx || exit
  make html
  cd .. || exit
fi

echo "-----------------------------------"
echo "Getting Docusaurus deps"
echo "-----------------------------------"
cd website || exit
yarn

if [[ $ONLY_DOCUSAURUS == false ]]; then
  # run script to parse html generated by sphinx
  echo "--------------------------------------------"
  echo "Parsing Sphinx docs and moving to Docusaurus"
  echo "--------------------------------------------"
  cd ..
  mkdir -p "website/pages/api/"

  cwd=$(pwd)
  python scripts/parse_sphinx.py -i "${cwd}/sphinx/build/html/" -o "${cwd}/website/pages/api/"

  SPHINX_JS_DIR='sphinx/build/html/_static/'
  DOCUSAURUS_JS_DIR='website/static/js/'

  mkdir -p $DOCUSAURUS_JS_DIR

  # move JS files from /sphinx/build/html/_static/*:
  cp "${SPHINX_JS_DIR}documentation_options.js" "${DOCUSAURUS_JS_DIR}documentation_options.js"
  cp "${SPHINX_JS_DIR}doctools.js" "${DOCUSAURUS_JS_DIR}doctools.js"
  cp "${SPHINX_JS_DIR}language_data.js" "${DOCUSAURUS_JS_DIR}language_data.js"
  cp "${SPHINX_JS_DIR}searchtools.js" "${DOCUSAURUS_JS_DIR}searchtools.js"

  # searchindex.js is not static util
  cp "sphinx/build/html/searchindex.js" "${DOCUSAURUS_JS_DIR}searchindex.js"

  # copy module sources
  cp -r "sphinx/build/html/_sources/" "website/static/_sphinx-sources/"

  echo "-----------------------------------"
  echo "Generating tutorials"
  echo "-----------------------------------"
  mkdir -p "website/_tutorials"
  mkdir -p "website/static/files"
  python scripts/parse_tutorials.py -w "${cwd}"

  cd website || exit
fi  # end of not only Docusaurus block

if [[ $BUILD_STATIC == true ]]; then
  echo "-----------------------------------"
  echo "Building static site"
  echo "-----------------------------------"
  yarn build
else
  echo "-----------------------------------"
  echo "Starting local server"
  echo "-----------------------------------"
  yarn start
fi
